import { createStore } from 'vuex'
import { v4 as uuidv4 } from "uuid"
import { instance, blob_instance, fixImageDuration, reorderArray } from "./axios"


export default createStore({
  state: {
    projects: [],
    files: [],
    clips: [],
    errors: [],
    uploads: [],
    exports: [],
    effects: [],
    project: null,
    current_export: { progress: 0.0 },
    user: null,
    preview: {
      file: null,
      clip: null,
      start: 0.0,
      end: 1.0,
      position: 0.0,
      length: 0.0
    },
    scrollToClip: null,
    default_image_length: 30
  },

  mutations: {
    addError(state, message) {
      if (typeof(message) == "object") {
        // Loop through object keys (potentially multiple errors)
        for (const key in message) {
          let error = {'id': uuidv4(), 'message': `${key}: ${message[key]}`}
          state.errors.push(error)
        }
      } else {
        // Single error message
        let error = {'id': uuidv4(), 'message': message}
        state.errors.push(error)
      }
    },
    removeError(state, error) {
      state.errors = state.errors.filter(err => err.id != error.id )
    },
    clearErrors(state) {
      state.errors = []
    },
    setScrollToClip(state, clipObj) {
      state.scrollToClip = clipObj
    },
    setUser(state, user) {
      state.user = user
    },
    addProject(state, projectObj) {
      state.projects.push(projectObj)
    },
    setProjects(state, projects) {
      state.projects = projects
    },
    setProject(state, project) {
      state.project = project
    },
    addEffect(state, effectObj) {
      state.effects.push(effectObj)
    },
    setEffects(state, effects) {
      state.effects = effects
    },
    setEffect(state, effectObj) {
      state.effects = [
        ...state.effects = state.effects.filter(e => e.id != effectObj.id),
        effectObj
      ]
    },
    updateProjectThumbnail(state, payload) {
      state.projects = [
        ...state.projects = state.projects.filter(p => p.id != payload.obj.id),
        payload.obj
      ]
    },
    deleteProject(state, project_id) {
      state.projects = state.projects.filter(p => p.id != project_id)
    },
    setFiles(state, files) {
      state.files = files
    },
    addFile(state, fileObj) {
      state.files.push(fileObj)
    },
    updateFileThumbnail(state, payload) {
      if (payload.obj.projects.includes(state.project.url)) {
        state.files = [
          ...state.files = state.files.filter(file => file.id != payload.obj.id),
          payload.obj
        ]
      }
    },
    deleteFile(state, file_id) {
      state.files = state.files.filter(file => file.id != file_id)
    },
    addClip(state, clip) {
      state.clips.push(clip)
    },
    setClips(state, clips) {
      state.clips = clips
      if (!clips) {
        // clear latest clip if no clips
        state.scrollToClip = null
      }
    },
    setClip(state, clipObj) {
      state.clips = [
        ...state.clips = state.clips.filter(clip => clip.id != clipObj.id),
        clipObj
      ]
    },
    deleteClip(state, clip_id) {
      state.clips = state.clips.filter(clip => clip.id != clip_id)
    },
    updateClipThumbnail(state, payload) {
      if (state.project.url == payload.obj.project) {
        state.clips = [
          ...state.clips = state.clips.filter(clip => clip.id != payload.obj.id),
          payload.obj
        ]
        if (payload.latest) {
          state.scrollToClip = payload.obj
        }
      }
    },
    setPreview(state, payload) {
      state.preview.start = payload.start
      state.preview.end = payload.end
      if (state.preview.file) {
        state.preview.length = (state.preview.end - state.preview.start) *
            fixImageDuration(state.preview.file.json.duration)
      }
    },
    setPreviewPosition(state, position) {
      state.preview.position = position
    },
    setPreviewFile(state, file) {
      state.preview.file = file
      state.preview.clip = null
      state.preview.start = 0.0
      state.preview.end = 1.0
      state.preview.position = 0.0
      if (state.preview.file) {
        state.preview.length = (state.preview.end - state.preview.start) *
            fixImageDuration(state.preview.file.json.duration)
      }
    },
    setPreviewClip(state, clipObj) {
      state.preview.clip = clipObj
      if (clipObj && clipObj.fileObj) {
        state.preview.start = clipObj.start / fixImageDuration(clipObj.fileObj.json.duration)
        state.preview.end = clipObj.end / fixImageDuration(clipObj.fileObj.json.duration)
        state.preview.position = clipObj.start / fixImageDuration(clipObj.fileObj.json.duration)
        if (state.preview.clip.fileObj) {
          state.preview.file = state.preview.clip.fileObj
        }
      } else {
        state.preview.file = null
        state.preview.start = 0.0
        state.preview.end = 1.0
        state.preview.position = 0.0
      }

      if (state.preview.file) {
        state.preview.length = (state.preview.end - state.preview.start) *
            fixImageDuration(state.preview.file.json.duration)
      }
    },
    addUpload(state, upload) {
      state.uploads.push(upload)
    },
    updateUpload(state, upload) {
      for (const up of state.uploads) {
        if (up.id == upload.id) {
          up.progress = upload.progress
          break
        }
      }
    },
    removeUpload(state, uploadID) {
      state.uploads = state.uploads.filter(up => up.id != uploadID )
    },
    setExports(state, exports) {
      state.exports = exports
      if (state.exports.length == 0) {
        state.current_export = { progress: 0.0 }
      }
    },
    setExport(state, exportObj) {
      state.exports = [
        ...state.exports = state.exports.filter(e => e.id != exportObj.id),
        exportObj
      ]
      state.current_export = exportObj
    },
  },
  getters: {
    isAuthenticated: state => !!state.user,
    totalClipDuration: state => {
      return state.clips.map(clip => clip.end - clip.start).reduce((prev, curr) => prev + curr, 0.0)
    },
    thumbnailedClips: state => {
      return state.clips.filter(clip => clip.thumbnail).sort((a, b) => a.position - b.position)
    },
    thumbnailedProjects: state => {
      return state.projects.filter(project => project.thumbnail).sort((a, b) => b.id - a.id)
    },
    activeExports: state => {
      return state.exports.filter(e => e.status == "preparing" ||
          e.status == "pending" ||
          e.status == "in-progress" ||
          e.status == "finalizing").sort((a, b) => b.id - a.id)
    }
  },
  actions: {
    async login({commit}, auth) {
      try {
        const response = await instance.get('/users/')
        let user_object = response.data.results[0]
        user_object.auth = auth
        commit('setUser', user_object)
      } catch(err) {
        commit('setUser', null)
        commit('addError', err.response.data)
      }
    },
    async loadProjects({commit, dispatch}) {
      try {
        const response = await instance.get('/projects/')
        // Sort all projects, and limit to X most recent
        let sorted_projects = response.data.results.sort().reverse().slice(0, 12)
        commit('setProjects', sorted_projects)
        for (let p of response.data.results) {
          let payload = { obj: p, frame: 1 }
          dispatch('attachThumbnail', payload)
        }
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async getProject({commit}, project_id) {
      try {
        return instance.get(`projects/${project_id}/`)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async createProject({commit, dispatch}, payload) {
      try {
        const response = await instance.post('/projects/', payload)
        commit('addProject', response)
        let thumbnail_payload = { obj: response.data, frame: 1 }
        dispatch('attachThumbnail', thumbnail_payload)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async updateProject({commit}, payload) {
      try {
        await instance.patch(`${payload.url}`, payload)
        commit('setProject', payload)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async deleteProject({commit}, project_id) {
      try {
        await instance.delete(`projects/${project_id}/`)
        commit('deleteProject', project_id)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async loadFiles({commit, dispatch}, project_id) {
      try {
        const response = await instance.get(`projects/${project_id}/files/`)
        let files = response.data.results

        // Generate thumbnail for each file object
        for (let fileObj of files) {
          // fix long duration on images
          fileObj.json.duration = fixImageDuration(fileObj.json.duration)
          let payload = { obj: fileObj, frame: 1 }
          dispatch('attachThumbnail', payload)
        }
        commit('setFiles', files)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async loadClips({commit, dispatch}, project_id) {
      try {
        const response = await instance.get(`projects/${project_id}/clips/`)
        let clips = response.data.results

        // Attach file obj to clip
        for (let clipObj of clips) {
          const file_response = await instance.get(clipObj.file)
          clipObj.fileObj = file_response.data
          commit('addClip', clipObj)

          let fps = clipObj.json.reader.fps.num / clipObj.json.reader.fps.den
          let thumbnail_payload = { obj: clipObj, frame: clipObj.start * fps, latest: false }
          dispatch('attachThumbnail', thumbnail_payload)
        }
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async createClip({dispatch, commit}, payload) {
      try {
        const response = await instance.post(`${payload.project_url}clips/`, payload.data)
        let clipObj = response.data
        const file_response = await instance.get(clipObj.file)
        clipObj.fileObj = file_response.data
        commit('addClip', clipObj)

        let fps = clipObj.json.reader.fps.num / clipObj.json.reader.fps.den
        let thumbnail_payload = { obj: clipObj, frame: clipObj.start * fps, latest: true }
        dispatch('attachThumbnail', thumbnail_payload)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async editClip({dispatch, commit}, payload) {
      try {
        await instance.patch(`${payload.data.url}`, payload.data)
        commit('setClip', payload.data)
        let fps = payload.data.json.reader.fps.num / payload.data.json.reader.fps.den
        let thumbnail_payload = { obj: payload.data, frame: payload.data.start * fps, latest: payload.latest }
        if (payload.thumbnail) {
          dispatch('attachThumbnail', thumbnail_payload)
        } else if (payload.latest) {
          commit('setScrollToClip', payload.data)
        }
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async moveClip({dispatch, getters}, payload) {
      let local_clips = [...getters.thumbnailedClips]
      let reordered_clips = local_clips
      if (payload.current != payload.dest) {
        reordered_clips = reorderArray(local_clips, payload.current, payload.dest)
      }
      let pos = 0.0
      for (let c of reordered_clips) {
        c.position = pos
        let edit_payload = { data: c, latest: false, thumbnail: false }
        if (c.id == payload.clip.id) {
          edit_payload.latest = true
        }
        dispatch('editClip', edit_payload)
        pos += (c.end - c.start)
      }
    },
    async deleteClip({commit}, clip_id) {
      try {
        await instance.delete(`clips/${clip_id}/`)
        commit('deleteClip', clip_id)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async deleteFile({commit}, file_id) {
      try {
        await instance.delete(`files/${file_id}/`)
        commit('deleteFile', file_id)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async createFile({commit, dispatch}, payload) {
      let uploadID = uuidv4()
      let uploadObject = { 'id': uploadID, progress: 0 }
      commit('addUpload', uploadObject)

      const config = {
        onUploadProgress: progressEvent => {
          let progress = (progressEvent.loaded / progressEvent.total) * 100;
          let uploadObject = { 'id': uploadID, progress }
          commit('updateUpload', uploadObject)
        }
      }

      try {
        let response = await instance.post(`${payload.project_url}files/`, payload.data, config)
        let fileObj = response.data
        // fix long duration on images
        fileObj.json.duration = fixImageDuration(fileObj.json.duration)
        commit('removeUpload', uploadID)

        let thumbnail_payload = { obj: fileObj, frame: 1 }
        dispatch('attachThumbnail', thumbnail_payload)
      } catch(err) {
        commit('removeUpload', uploadID)
        commit('addError', err.response.data)
      }
    },
    async attachThumbnail({commit}, payload) {
      let Obj = payload.obj
      let objectType = null
      if (payload.obj.file) {
        // Clip passed in
        Obj = payload.obj.fileObj
        objectType = 'clip'
      } else if (payload.obj.media) {
        // File passed in
        objectType = 'file'
      } else {
        // Project passed in
        objectType = 'project'
      }
      let sessionKey = `${objectType}-${Obj.id}-frame${payload.frame}`

      if (payload.clobber) {
        // Destroy existing cached thumbnail
        sessionStorage.removeItem(sessionKey)
      }

      let data = {
        "frame_number": payload.frame,
        "width": 480,
        "height": 270,
        "image_format": "JPEG",
        "image_quality": 100
      }
      try {
        let thumbnailUrl = sessionStorage.getItem(sessionKey)
        if (thumbnailUrl) {
          try {
            // Verify blob
            await blob_instance.get(thumbnailUrl)
          } catch(err) {
            // Failed to find blob
            thumbnailUrl = null
          }
        }
        if (!thumbnailUrl) {
          const response = await instance.post(`${Obj.url}thumbnail/`, data, { responseType: 'arraybuffer' })
          let blob = new Blob(
              [response.data],
              { type: response.headers['content-type'] }
          )
          thumbnailUrl = URL.createObjectURL(blob)
          sessionStorage.setItem(sessionKey, thumbnailUrl)
        }

        // Append thumbnail attribute to each file object
        payload.obj.thumbnail = thumbnailUrl
        if (objectType == 'file') {
          commit('updateFileThumbnail', payload)
        } else if (objectType == 'clip') {
          commit('updateClipThumbnail', payload)
        } else {
          commit('updateProjectThumbnail', payload)
        }

      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async createExport({commit}, payload) {
      try {
        const response = await instance.post('/exports/', payload)
        commit('setExport', response.data)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async loadExports({commit}, project_id) {
      try {
        const proj_response = await instance.get(`projects/${project_id}/`)
        const sorted_export_urls = proj_response.data.exports.sort().reverse()
        if (sorted_export_urls.length > 0) {
          // Only get most recent Export record
          const export_response = await instance.get(sorted_export_urls[0])
          commit('setExport', export_response.data)
        }
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async checkExportProgress({commit}, exportObj) {
      try {
        const export_response = await instance.get(exportObj.url)
        commit('setExport', export_response.data)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async loadEffects({commit}, project_id) {
      try {
        const effects_response = await instance.get(`projects/${project_id}/effects/`)
        commit('setEffects', effects_response.data.results)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async createEffect({commit}, payload) {
      try {
        const response = await instance.post('/effects/', payload)
        commit('addEffect', response.data)
      } catch(err) {
        commit('addError', err.response.data)
      }
    },
    async updateEffect({commit}, payload) {
      try {
        await instance.patch(payload.url, payload)
        commit('setEffect', payload)
      } catch(err) {
        console.log(err)
        commit('addError', err.response.data)
      }
    },
  },
  modules: {
  }
})
