import { v4 as uuidv4 } from "uuid"
import captionTemplate from "../data/caption.json"

export const CAPTION_POSITION_MAP = {
  top: 0.1,
  center: 0.45,
  bottom: 0.75
}

export const DEFAULT_CAPTION_POSITION = 'bottom'

const CAPTION_KEYWORDS = ['caption']

function deepClone(value) {
  return JSON.parse(JSON.stringify(value))
}

function padNumber(value, size) {
  const normalized = Math.max(0, Math.floor(Math.abs(value)))
  return normalized.toString().padStart(size, '0')
}

function secondsToTimestamp(seconds) {
  const normalized = Number.isFinite(seconds) ? Math.max(0, seconds) : 0
  const totalMilliseconds = Math.round(normalized * 1000)
  const hours = Math.floor(totalMilliseconds / 3600000)
  const minutes = Math.floor((totalMilliseconds % 3600000) / 60000)
  const secs = Math.floor((totalMilliseconds % 60000) / 1000)
  const millis = totalMilliseconds % 1000
  return `${padNumber(hours, 2)}:${padNumber(minutes, 2)}:${padNumber(secs, 2)}:${padNumber(millis, 3)}`
}

function buildCaptionTimestampText(clip) {
  const start = Number.isFinite(clip?.start) ? clip.start : 0
  let end = Number.isFinite(clip?.end) ? clip.end : start
  if (end < start) {
    end = start
  }
  const startText = secondsToTimestamp(start)
  const endText = secondsToTimestamp(end)
  return { start, end, startText, endText }
}

function extractCaptionBody(effect) {
  if (!effect || typeof effect.caption_text !== 'string') {
    return ""
  }
  const sanitized = effect.caption_text.replace(/\r/g, '')
  const newlineIndex = sanitized.indexOf('\n')
  if (newlineIndex === -1) {
    return sanitized.trim()
  }
  return sanitized.slice(newlineIndex + 1).replace(/\n+$/g, '').trim()
}

export function isCaptionEffect(effect) {
  if (!effect) {
    return false
  }
  const identifiers = [
    effect.class_name,
    effect.type,
    effect.title,
    effect.name,
    effect.display_name
  ]
  return identifiers.some(identifier => {
    if (typeof identifier !== 'string') {
      return false
    }
    const normalized = identifier.toLowerCase()
    return CAPTION_KEYWORDS.some(keyword => normalized.includes(keyword))
  })
}

function getClipEffects(clip) {
  if (Array.isArray(clip?.json?.effects)) {
    return clip.json.effects
  }
  return []
}

function findCaptionEffectInEffects(effects) {
  if (!Array.isArray(effects) || effects.length === 0) {
    return { effect: null, index: -1 }
  }
  const index = effects.findIndex(effect => isCaptionEffect(effect))
  if (index === -1) {
    return { effect: null, index: -1 }
  }
  return { effect: effects[index], index }
}

function findCaptionEffect(clip) {
  const effects = getClipEffects(clip)
  return findCaptionEffectInEffects(effects)
}

function getEffectTopValue(effect) {
  if (!effect || !effect.top || !Array.isArray(effect.top.Points) || effect.top.Points.length === 0) {
    return null
  }
  const point = effect.top.Points[0]
  if (point && point.co && typeof point.co.Y === 'number') {
    return point.co.Y
  }
  return null
}

function ensureEffectTop(effect, topValue) {
  if (!effect.top || !Array.isArray(effect.top.Points) || effect.top.Points.length === 0) {
    if (captionTemplate?.top && Array.isArray(captionTemplate.top.Points)) {
      effect.top = deepClone(captionTemplate.top)
    } else {
      effect.top = {
        Points: [
          {
            co: { X: 1, Y: captionPositionToTop(DEFAULT_CAPTION_POSITION) },
            handle_type: 0,
            interpolation: 0
          }
        ]
      }
    }
  }
  if (!effect.top.Points[0].co) {
    effect.top.Points[0].co = { X: 1, Y: captionPositionToTop(DEFAULT_CAPTION_POSITION) }
  }
  effect.top.Points[0].co.Y = topValue
}

function ensureEffectIdentity(effect) {
  if (typeof effect.id !== 'string' || !effect.id.length) {
    effect.id = uuidv4().split('-')[0]
  }
}

function applyEffectTiming(effect, timing) {
  effect.position = 0
  effect.start = timing.start
  effect.end = timing.end
  effect.duration = Math.max(0, timing.end - timing.start)
}

function buildCaptionEffect(baseEffect, textValue, clip, normalizedPosition) {
  const effect = baseEffect ? deepClone(baseEffect) : deepClone(captionTemplate)
  ensureEffectIdentity(effect)
  const timing = buildCaptionTimestampText(clip)
  applyEffectTiming(effect, timing)
  effect.caption_text = `${timing.startText} --> ${timing.endText}\n${textValue}`
  effect.apply_before_clip = false
  ensureEffectTop(effect, captionPositionToTop(normalizedPosition))
  return effect
}

export function normalizeCaptionPosition(position) {
  if (typeof position !== 'string') {
    return DEFAULT_CAPTION_POSITION
  }
  const normalized = position.toLowerCase()
  if (Object.prototype.hasOwnProperty.call(CAPTION_POSITION_MAP, normalized)) {
    return normalized
  }
  return DEFAULT_CAPTION_POSITION
}

export function captionPositionToTop(position) {
  const normalized = normalizeCaptionPosition(position)
  return CAPTION_POSITION_MAP[normalized]
}

export function topValueToCaptionPosition(value) {
  if (typeof value !== 'number' || !Number.isFinite(value)) {
    return DEFAULT_CAPTION_POSITION
  }
  let closestKey = DEFAULT_CAPTION_POSITION
  let smallestDelta = Infinity
  Object.entries(CAPTION_POSITION_MAP).forEach(([key, topValue]) => {
    const delta = Math.abs(topValue - value)
    if (delta < smallestDelta) {
      smallestDelta = delta
      closestKey = key
    }
  })
  return closestKey
}

export function getClipCaptionState(clip) {
  const { effect } = findCaptionEffect(clip)
  const effectText = extractCaptionBody(effect)
  const legacyText = clip?.json?.text
  const text = effectText || (typeof legacyText === 'string' ? legacyText : "")
  const effectTop = getEffectTopValue(effect)
  const storedTop = typeof clip?.json?.caption_top === 'number' ? clip.json.caption_top : null
  const storedPosition = clip?.json?.caption_position
  let normalizedPosition = DEFAULT_CAPTION_POSITION
  let captionTop = captionPositionToTop(normalizedPosition)
  if (typeof effectTop === 'number') {
    captionTop = effectTop
    normalizedPosition = topValueToCaptionPosition(effectTop)
  } else if (typeof storedTop === 'number') {
    captionTop = storedTop
    normalizedPosition = topValueToCaptionPosition(storedTop)
  } else if (storedPosition) {
    normalizedPosition = normalizeCaptionPosition(storedPosition)
    captionTop = captionPositionToTop(normalizedPosition)
  }
  return {
    text,
    position: normalizedPosition,
    top: captionTop,
    hasCaptionEffect: !!effect
  }
}

export function createClipWithCaptionForm(clip, form, options = {}) {
  if (!clip) {
    return null
  }
  const normalizedPosition = normalizeCaptionPosition(form?.position)
  const textValue = form?.text ?? ""
  const nextClip = {
    ...clip,
    json: {
      ...(clip.json || {})
    }
  }
  const existingEffects = Array.isArray(clip?.json?.effects)
    ? clip.json.effects.map(effect => deepClone(effect))
    : []
  const { index } = findCaptionEffectInEffects(existingEffects)
  const hasText = typeof textValue === 'string' && textValue.length > 0
  if (hasText) {
    const existingEffect = index >= 0 ? existingEffects[index] : null
    const updatedEffect = buildCaptionEffect(existingEffect, textValue, nextClip, normalizedPosition)
    if (index >= 0) {
      existingEffects[index] = updatedEffect
    } else {
      existingEffects.push(updatedEffect)
    }
  } else if (index >= 0) {
    existingEffects.splice(index, 1)
  }
  if (existingEffects.length > 0) {
    nextClip.json.effects = existingEffects
  } else {
    delete nextClip.json.effects
  }
  delete nextClip.json.text
  delete nextClip.json.caption_top
  delete nextClip.json.caption_position
  nextClip.json.fx_enabled = !!options.fxEnabled
  return nextClip
}

export function syncCaptionEffectWithClip(clip) {
  if (!clip || !clip.json || !Array.isArray(clip.json.effects)) {
    return clip
  }
  const { index } = findCaptionEffect(clip)
  if (index === -1) {
    return clip
  }
  const state = getClipCaptionState(clip)
  if (!state.text) {
    clip.json.effects.splice(index, 1)
    if (clip.json.effects.length === 0) {
      delete clip.json.effects
    }
    return clip
  }
  const updatedEffect = buildCaptionEffect(clip.json.effects[index], state.text, clip, state.position)
  clip.json.effects[index] = updatedEffect
  return clip
}

export function hasCaptionChanges(clip, form) {
  if (!clip) {
    return false
  }
  const currentState = getClipCaptionState(clip)
  const desiredText = form?.text ?? ""
  const desiredPosition = normalizeCaptionPosition(form?.position)
  const wantsEffect = !!desiredText
  const hasEffect = !!currentState.hasCaptionEffect
  return currentState.text !== desiredText
    || currentState.position !== desiredPosition
    || wantsEffect !== hasEffect
}
