const ADS_TAG_ID = process.env.VUE_APP_GOOGLE_ADS_ID || 'AW-994591350'

const CONVERSIONS = {
  demo: 'AW-994591350/fp_tCPiIj6AYEPaEodoD',
  aws: 'AW-994591350/BoJoCJ6634IBEPaEodoD',
  azure: 'AW-994591350/MREMCM28jqAYEPaEodoD',
  google: 'AW-994591350/W6AWCOyvgaAYEPaEodoD'
}

export function isDemoEnvironment() {
  const apiUrl = (process.env.VUE_APP_OPENSHOT_API_URL || '').trim()
  if (!apiUrl) {
    return false
  }
  const normalized = apiUrl.replace(/\/+$/, '').toLowerCase()
  return normalized === 'https://cloud.openshot.org'
}

function hasGtag() {
  return typeof window !== 'undefined' && typeof window.gtag === 'function'
}

function ensureAdsTag() {
  if (!ADS_TAG_ID || !hasGtag()) {
    return
  }
  window.gtag('config', ADS_TAG_ID)
}

function reportConversion(conversionKey) {
  const conversionId = CONVERSIONS[conversionKey]
  if (!conversionId || !hasGtag()) {
    return false
  }
  window.gtag('event', 'conversion', {
    send_to: conversionId,
    event_callback: () => {}
  })
  return true
}

function resolveConversionKey(target) {
  if (!target || !target.classList) {
    return null
  }
  const dataKey = target.dataset.adsConversion
  if (dataKey) {
    return dataKey
  }
  if (target.classList.contains('js-ads-aws')) {
    return 'aws'
  }
  if (target.classList.contains('js-ads-azure')) {
    return 'azure'
  }
  if (target.classList.contains('js-ads-gcp')) {
    return 'google'
  }
  if (target.classList.contains('js-ads-demo')) {
    return 'demo'
  }
  return null
}

function registerClickTracking() {
  document.addEventListener('click', (event) => {
    const target = event.target && event.target.closest
      ? event.target.closest('a, button')
      : null
    if (!target) {
      return
    }
    const conversionKey = resolveConversionKey(target)
    if (!conversionKey) {
      return
    }
    reportConversion(conversionKey)
  })
}

function registerDemoOnRoutes(router) {
  const fireDemo = () => reportConversion('demo')
  fireDemo()
  if (router && typeof router.afterEach === 'function') {
    router.afterEach(() => {
      fireDemo()
    })
  }
}

export function initAdsConversions({ router, demoMode }) {
  if (!demoMode) {
    return
  }
  ensureAdsTag()
  registerClickTracking()
  registerDemoOnRoutes(router)
}
